<?php
session_start();
include 'conexion.php';

// Configurar zona horaria
date_default_timezone_set('America/Santiago');

// Verificar si el usuario tiene rol de conserje
if (!isset($_SESSION['rol']) || $_SESSION['rol'] !== 'conserje') {
    header("Location: login.php");
    exit;
}

// Obtener parámetros de búsqueda
$busqueda = isset($_GET['busqueda']) ? $_GET['busqueda'] : '';

// Consultar arrendatarios con búsqueda dinámica
$query = "
    SELECT a.*, r.fecha_hora_llegada, r.fecha_hora_salida, p.numero_departamento, u.nombre AS propietario
    FROM arrendatarios a
    LEFT JOIN registros r ON a.id_arrendatario = r.id_arrendatario
    JOIN propietarios p ON a.id_propietario = p.id_propietario
    JOIN usuarios u ON p.id_usuario = u.id_usuario
    WHERE a.nombre LIKE ? OR a.rut LIKE ? OR p.numero_departamento LIKE ?
    ORDER BY a.fecha_inicio ASC;
";

$stmt = $conn->prepare($query);
$busqueda_param = '%' . $busqueda . '%';
$stmt->bind_param("sss", $busqueda_param, $busqueda_param, $busqueda_param);
$stmt->execute();
$result = $stmt->get_result();

function calcularEstado($fechaInicio, $fechaSalida) {
    $fechaActual = date("Y-m-d");
    if ($fechaActual < $fechaInicio) {
        return "Pendiente";
    } elseif ($fechaActual >= $fechaInicio && $fechaActual <= $fechaSalida) {
        return "En Arriendo";
    } else {
        return "Finalizado";
    }
}

function calcularDiasArriendo($fechaInicio, $fechaSalida) {
    $datetime1 = new DateTime($fechaInicio);
    $datetime2 = new DateTime($fechaSalida);
    $interval = $datetime1->diff($datetime2);
    return $interval->days + 1;
}

// Preparar eventos para el calendario
$events = [];
while ($row = $result->fetch_assoc()) {
    $estado = calcularEstado($row['fecha_inicio'], $row['fecha_salida']);
    if ($estado !== "En Arriendo") continue;

    $events[] = [
        'title' => "Ingreso: " . $row['nombre'],
        'start' => $row['fecha_inicio'],
        'color' => 'green'
    ];
    $events[] = [
        'title' => "Salida: " . $row['nombre'],
        'start' => $row['fecha_salida'],
        'color' => 'red'
    ];
}

// Reiniciar el resultado para usarlo en la tabla
$stmt->execute();
$result = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard Conserje</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@fortawesome/fontawesome-free/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.8/main.min.css">
</head>
<body class="hold-transition sidebar-mini">
<div class="wrapper">
    <!-- Navbar -->
    <nav class="main-header navbar navbar-expand navbar-white navbar-light">
        <ul class="navbar-nav">
            <li class="nav-item">
                <a class="nav-link" data-widget="pushmenu" href="#" role="button"><i class="fas fa-bars"></i></a>
            </li>
        </ul>
        <ul class="navbar-nav ml-auto">
            <li class="nav-item">
                <a class="nav-link" href="login.php" role="button">
                    <i class="fas fa-sign-out-alt"></i> Cerrar Sesión
                </a>
            </li>
        </ul>
    </nav>

    <!-- Sidebar -->
    <aside class="main-sidebar sidebar-dark-primary elevation-4">
        <a href="#" class="brand-link">
            <span class="brand-text font-weight-light">Sistema Arriendos</span>
        </a>
        <div class="sidebar">
            <div class="user-panel mt-3 pb-3 mb-3 d-flex">
                <div class="info">
                    <a href="#" class="d-block">Conserje</a>
                </div>
            </div>
            <nav class="mt-2">
                <ul class="nav nav-pills nav-sidebar flex-column" data-widget="treeview" role="menu">
                    <li class="nav-item">
                        <a href="conserje_dashboard.php" class="nav-link active">
                            <i class="nav-icon fas fa-home"></i>
                            <p>Dashboard</p>
                        </a>
                    </li>
                </ul>
            </nav>
        </div>
    </aside>

    <!-- Content Wrapper -->
    <div class="content-wrapper">
        <section class="content-header">
            <div class="container-fluid">
                <div class="row mb-2">
                    <div class="col-sm-6">
                        <h1>Lista de Arrendatarios</h1>
                    </div>
                    <div class="col-sm-6">
                        <!-- Campo de búsqueda -->
                        <form method="GET" class="d-flex">
                            <input type="text" name="busqueda" class="form-control me-2" placeholder="Buscar por nombre o RUT" value="<?= htmlspecialchars($busqueda); ?>">
                            <button type="submit" class="btn btn-primary">Buscar</button>
                        </form>
                    </div>
                </div>
            </div>
        </section>

        <!-- Calendario -->
        <section class="content">
            <div class="container-fluid mb-3">
                <div id="calendar"></div>
            </div>
        </section>

        <section class="content">
            <div class="container-fluid">
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th>Nombre Arrendatario</th>
                            <th>RUT</th>
                            <th>Teléfono</th>
                            <th>Departamento</th>
                            <th>Propietario</th>
                            <th>Fecha de Inicio</th>
                            <th>Fecha de Salida</th>
                            <th>Estado</th>
                            <th>Registro Llegada</th>
                            <th>Registro Salida</th>
                            <th>Ver Ficha</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($row = $result->fetch_assoc()): 
                            $estado = calcularEstado($row['fecha_inicio'], $row['fecha_salida']);
                            if ($estado !== "En Arriendo") continue;

                            $diasArriendo = calcularDiasArriendo($row['fecha_inicio'], $row['fecha_salida']);
                            $fechaSalida = $row['fecha_salida'];
                            $fechaInicio = $row['fecha_inicio'];
                            $fechaActual = date("Y-m-d");

                            $mostrarSpinnerIngreso = ($fechaInicio <= $fechaActual && !$row['fecha_hora_llegada']);
                            $mostrarSpinnerSalida = ($fechaSalida <= $fechaActual && !$row['fecha_hora_salida']);
                        ?>
                        <tr>
                            <td><?= htmlspecialchars($row['nombre']); ?></td>
                            <td><?= htmlspecialchars($row['rut']); ?></td>
                            <td><?= htmlspecialchars($row['celular']); ?></td>
                            <td><?= htmlspecialchars($row['numero_departamento']); ?></td>
                            <td><?= htmlspecialchars($row['propietario']); ?></td>
                            <td><?= date("d-m-Y", strtotime($row['fecha_inicio'])); ?></td>
                            <td><?= date("d-m-Y", strtotime($row['fecha_salida'])); ?></td>
                            <td><span class="badge bg-success"><?= $estado; ?></span></td>
                            <td>
                                <?= $row['fecha_hora_llegada'] 
                                    ? date("d-m-Y H:i", strtotime($row['fecha_hora_llegada'])) 
                                    : 'Pendiente'; ?>
                                <?php if ($mostrarSpinnerIngreso): ?>
                                    <div class="spinner-grow text-success" role="status" style="width: 20px; height: 20px;" title="Llegada pendiente"></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?= $row['fecha_hora_salida'] 
                                    ? date("d-m-Y H:i", strtotime($row['fecha_hora_salida'])) 
                                    : 'Pendiente'; ?>
                                <?php if ($mostrarSpinnerSalida): ?>
                                    <div class="spinner-grow text-danger" role="status" style="width: 20px; height: 20px;" title="Salida atrasada o pendiente"></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="generar_ficha.php?id=<?= $row['id_arrendatario']; ?>&dias=<?= $diasArriendo; ?>" class="btn btn-info" target="_blank">Ver Ficha</a>
                            </td>
                            <td>
                                <form action="registrar_evento.php" method="POST">
                                    <input type="hidden" name="id_arrendatario" value="<?= $row['id_arrendatario']; ?>">
                                    <?php if (!$row['fecha_hora_llegada']): ?>
                                        <button type="submit" name="registrar_llegada" class="btn btn-success">Registrar Llegada</button>
                                    <?php elseif (!$row['fecha_hora_salida']): ?>
                                        <button type="submit" name="registrar_salida" class="btn btn-danger">Registrar Salida</button>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Completado</span>
                                    <?php endif; ?>
                                </form>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </section>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.8/main.min.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        var calendarEl = document.getElementById('calendar');
        var calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,timeGridDay'
            },
            events: <?= json_encode($events); ?>
        });
        calendar.render();
    });
</script>
</body>
</html>
